import { ModalFormData } from "@minecraft/server-ui";
import { SchoolSubject } from "./Classes";
import { School } from "./School";
/**
 * @description Represents a Human in the world
 */
class Human{
    /**
     * Represents a Human in the world
     * @param {string} name This Human's name
     * @param {string} fname This Human's family name
     * @param {Date} birthdate This Human's birthdate
     * @param {string|number} id This Human's identifier
     */
    constructor(name, fname, birthdate, id){
        /**
         * @type {string}
         * @description This Human's name
         */
        this.name = name;
        /**
         * @type {string}
         * @description This Human's family name
         */
        this.familyName = fname;
        /**
         * @type {Date}
         * @description This Human's birth date (first time log-on)
         */
        this.birthdate = birthdate;
        /**
         * @type {Array<string>}
         * @description Roles and groups this Human is part of
         */
        this.roles = [];
        /**
         * @type {string}
         * @description This Human's gender
         */
        this.gender = "";
        /**
         * @type {number|string}
         * @description This Human's unique identifier
         */
        this.identifier = id;
        /**
         * @type {number}
         * @description This Human's Human-Type (Human 0, Teacher 1, ...)
         */
        this.type = 0;

        /**
         * @type {boolean}
         * @description Is this Human married?
         */
        this.isMarried = false;
        /**
         * @type {Array<string|number>}
         * @description Humans (identifiers) who this Human loves
         */
        this.loves = [];
        /**
         * @type {Array<string|number>}
         * @description Humans (identifiers) who this Human hates
         */
        this.hates = [];
        /**
         * @type {Array<string|number>}
         * @description Humans (identifiers) who count as this Human's children
         */
        this.parentOf = [];
        /**
         * @type {Array<string|number>}
         * @description Humans (identifiers) who count as this Human's parents
         */
        this.childOf = [];
    };
    /**
     * Returns a string with Human data
     * @returns {string}
     */
    getSaveString(){
        let saved = {
            name: this.name, familyName:this.familyName, birthdate:this.birthdate, roles:this.roles, gender:this.gender, identifier:this.identifier, type:this.type,
            isMarried: this.isMarried, loves: this.loves, hates: this.hates, parentOf: this.parentOf, childOf: this.childOf
        }; // ^ old format
        let {...data} = this;
        return JSON.stringify(data);
    };
    /**
     * Loads data from a saved string retrieved from Human.getSaveString()
     * @param {string} saved The string with Human data to load. Every data is required!
     */
    loadSaveString(saved){
        /**
         * @type {Human}
         */
        let data = JSON.parse(saved);
        this.name = data.name;
        this.familyName = data.familyName;
        this.birthdate = data.birthdate;
        this.roles = data.roles;
        this.gender = data.gender;
        this.identifier = data.identifier;
        //this.type = data.type;

        this.isMarried = data.isMarried;
        this.loves = data.loves;
        this.hates = data.hates;
        this.parentOf = data.parentOf;
        this.childOf = data.childOf;
    };
    /**
     * @type {Array<Human|Teacher|Student>}
     * @description List of all registered Human beeings
     */
    static humanList = [];
    registerHuman(){
        for(let h = 0; h < Human.humanList.length; h++){ if(Human.humanList[h].identifier==this.identifier){ console.warn("Human with identifier: "+this.identifier+", already registered!"); return; } };
        Human.humanList.push(this);
    };
    /**
     * Search for a registered human.
     * @param {string} id The human's identifier to be looking for.
     * @returns {Human|Teacher|Student|undefined} The human with that identifier.
     */
    static lookForHuman(id){
        for(let s = 0; s < Human.humanList.length; s++){ if(Human.humanList[s].identifier == id){ return Human.humanList[s]; }; };
        console.error("Human not found! Id searched: "+id);
        return undefined
    };
    /**
     * Turns this Human into a Teacher
     * @returns {Teacher|void}
     */
    becomeTeacher(){
        if(this.type == 1){console.warn("Cannot become a Teacher! Already a Teacher."); return;}else 
        if(this.type == 0 || this.type == 2){
            let tO = new Teacher(this.name, this.familyName, this.birthdate, this.identifier); tO.loadSaveString(this.getSaveString()); for(let h = 0; h<Human.humanList.length; h++){if(Human.humanList[h].identifier==tO.identifier){Human.humanList[h]=tO}}; console.warn("Human became Teacher! Id: "+this.identifier); return tO;
        };
    };
    /**
     * Turns this Human into a Student
     * @returns {Student|void}
     */
    becomeStudent(){
        if(this.type == 2){console.warn("Cannot become a Student! Already a Student."); return;}else 
        if(this.type == 0 || this.type == 1){
            let sO = new Student(this.name, this.familyName, this.birthdate, this.identifier); sO.loadSaveString(this.getSaveString()); for(let h = 0; h<Human.humanList.length; h++){if(Human.humanList[h].identifier==sO.identifier){Human.humanList[h]=sO}}; console.warn("Human became Student! Id: "+this.identifier); return sO;
        };
    };
};
/**
 * @description Represents a Teacher in the world
 */
class Teacher extends Human{
    /**
     * Represents a Teacher in the world
     * @param {string} name This Teacher's name
     * @param {string} fname This Teacher's family name
     * @param {Date} bday This Teacher's birthdate
     * @param {string|number} id This Teacher's identifier
     */
    constructor(name, fname, bday, id){
        super(name, fname, bday, id);
        /**
         * @type {Array<SchoolSubject>}
         * @description The subjects this Teacher teaches
         */
        this.subjects = [];
        /**
         * @type {Array<number>}
         * @description What grades does this teacher teach?
         */
        this.gradeTeaching = [];

        this.type=1;
    }
    /**
     * Loads data from a Teacher data string (retrieved from Teacher.getSaveTeacherString())
     * @param {string} data String holding Teacher data
     */
    loadDataFromTeacher(data){
        /**
         * @type {Teacher}
         */
        let d = JSON.parse(data);
        this.name = d.name;
        this.familyName = d.familyName;
        this.birthdate = d.birthdate;
        this.roles = d.roles;
        this.gender = d.gender;
        this.identifier = d.identifier;

        this.isMarried = d.isMarried;
        this.loves = d.loves;
        this.hates = d.hates;
        this.parentOf = d.parentOf;
        this.childOf = d.childOf;

        this.subjects = d.subjects;
        this.gradeTeaching = d.gradeTeaching;
    };
    /**
     * Returns a string with Teacher data
     * @returns {string}
     */
    getSaveTeacherString(){
        let {...data} = this;
        return JSON.stringify(data);
    };
    /**
     * Turns this Teacher into a Human
     * @returns {Human|void}
     */
    becomeHuman(){
        let hO = new Human(this.name, this.familyName, this.birthdate, this.identifier); hO.loadSaveString(this.getSaveString()); for(let h = 0; h<Human.humanList.length; h++){if(Human.humanList[h].identifier==this.identifier){Human.humanList[h]=hO;}}; console.warn("Teacher became Human! Id: "+this.identifier); return hO;
    };
};
/**
 * @description Represents a Student in the world
 */
class Student extends Human{
    /**
     * @description Represents a Student in the world
     * @param {string} name This student's name
     * @param {string} fname This student's family name
     * @param {Date} bday This student's birthdate
     * @param {string|number} id This student's id
     */
    constructor(name, fname, bday, id){
        super(name, fname, bday, id);
        /**
         * @type {Array<string>}
         * @description School Subjects this Student is good at
         */
        this.goodSubjects = [];
        /**
         * @type {Array<string>}
         * @description School Subjects this Student is bad at
         */
        this.badSubjects = [];
        /**
         * @type {number}
         * @description This Student's intelligence
         */
        this.intelligence = NaN;
        /**
         * @type {number}
         * @description The grade this Student is at
         */
        this.grade = NaN;
        /**
         * @type {number}
         * @description The learn percentage for this Student (how much has this student recently learned)
         */
        this.learnPercentage = NaN;

        this.type=2;
    };
    /**
     * Returns a percentage (%) this student could have archieved in this test.
     * @param {SchoolSubject} subject The subject this test is about
     * @returns {number} A percent score
     */
    writeTest(subject){
        let skillMul = 5; let intelliMul; let learnMul; let multiplier;
        if(this.goodSubjects.includes(subject)==true){ skillMul = Math.random()*20+this.intelligence/2; }else if(this.badSubjects.includes(subject)==true){ skillMul = Math.random()*-10-10; }else {skillMul = 5};
        if(Number.isNaN(this.intelligence)==true){ intelliMul = 1; }else { intelliMul = Math.round(this.intelligence/5); };
        if(Number.isNaN(this.learnPercentage)==true){ learnMul = 1; }else { learnMul = Math.round(this.learnPercentage/3); };
        multiplier = skillMul+intelliMul+learnMul;
        return multiplier;
    };
    /**
     * Loads data from a Student data string (retrieved from Student.getSaveStudentString())
     * @param {string} data String holding Student data
     */
    loadDataFromStudent(data){
        /**
         * @type {Student}
         */
        let d = JSON.parse(data); this.name = d.name; this.familyName = d.familyName; this.birthdate = d.birthdate; this.roles = d.roles; this.gender = d.gender; this.identifier = d.identifier;
        this.isMarried = d.isMarried; this.loves = d.loves; this.hates = d.hates; this.parentOf = d.parentOf; this.childOf = d.childOf;
        if(d.badSubjects){this.badSubjects=d.badSubjects}; if(d.goodSubjects){this.goodSubjects = d.goodSubjects}; if(d.intelligence){this.intelligence=d.intelligence}; if(d.grade){this.grade=d.grade}; if(d.learnPercentage){this.learnPercentage=d.learnPercentage};
    };
    /**
     * Returns a string with Student data
     * @returns {string}
     */
    getSaveStudentString(){
        let {...data} = this;
        console.warn(JSON.stringify(data));
        return JSON.stringify(data);
    };
    /**
     * Turns this Student into a Human
     * @returns {Human|void}
     */
    becomeHuman(){
        let hO = new Human(this.name, this.familyName, this.birthdate, this.identifier); hO.loadSaveString(this.getSaveString()); for(let h = 0; h<Human.humanList.length; h++){if(Human.humanList[h].identifier==this.identifier){Human.humanList[h]=hO;}}; console.warn("Student became Human! Id: "+this.identifier); return hO;
    };
};

export {Human, Student, Teacher};
/**
 * Adds a Human as a Personal
 * @param {Player} player The player to add Personal
 * @param {School} schol The school
 */
export function addPersonalToClass(player, schol){
    let main = new ModalFormData(); main.title("School"); 
    let op = [];
    if(Human.humanList.length!=0){
        for(let h =0; h<Human.humanList.length; h++){op.push(Human.humanList[h].name+", "+Human.humanList[h].identifier)};
        main.dropdown("Select Human to add as School Personal", op,0);
    }else {player.sendMessage("There are no Registered Humans! try reloading");return;};
    main.show(player).then((response)=>{
        if(response.canceled){return;};schol.personal.push(Human.lookForHuman(Human.humanList[response.formValues[0]].identifier));
    })
}